/* linebuffer.c -- read arbitrarily long lines
   Copyright (C) 1986, 1991, 1998, 1999 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Written by Richard Stallman and Padraig Brady */


#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include "unlocked-io.h"
#include "linebuffer.h"

void free ();

/* Should really allow this to be passed to initbuffer */
#define INITIAL_LINE_LENGTH 200

/* Initialize linebuffer LINEBUFFER for use. */
void
initbuffer (struct linebuffer *linebuffer)
{
  linebuffer->length = 0;
  linebuffer->size = INITIAL_LINE_LENGTH;
  linebuffer->buffer = (char *) malloc (linebuffer->size);
}

/* Read an arbitrarily long line of text from STREAM into LINEBUFFER.
   Keep the newline; append a newline if it's the last line of a file
   that ends in a non-newline character. Do not null terminate.
   Therefore the stream can contain NULs and the length (including
   the newline) is returned in linebuffer->length.
   Return LINEBUFFER, except at end of file or error return NULL.*/
struct linebuffer *
readline (struct linebuffer *linebuffer, FILE *stream)
{
#if GETLINE
  ssize_t tmp_len;
  if ((tmp_len=getline(&linebuffer->buffer, &linebuffer->size, stream)) <= 0)
    return NULL;
  linebuffer->length=tmp_len;
  if (linebuffer->buffer[tmp_len-1] != '\n')
    {
      linebuffer->buffer[tmp_len] = '\n';
      linebuffer->length++;
    }
#elif (HAVE_GETC_UNLOCKED && !FORCE_FGETS) || FORCE_GETC
  int c;
  char *buffer = linebuffer->buffer;
  char *p = linebuffer->buffer;
  char *end = buffer + linebuffer->size; /* Sentinel. */

  if (feof (stream) || ferror (stream))
    return 0;

  do
    {
      c = getc (stream);
      if (c == EOF)
	{
	  if (p == buffer)
	    return 0;
	  if (p[-1] == '\n')
	    break;
	  c = '\n';
	}
      if (p == end)
	{
	  linebuffer->size *= 2;
	  buffer = (char *) realloc (buffer, linebuffer->size);
	  p = p - linebuffer->buffer + buffer;
	  linebuffer->buffer = buffer;
	  end = buffer + linebuffer->size;
	}
      *p++ = c;
    }
  while (c != '\n');

  linebuffer->length = p - buffer;
#else
  size_t fgetSize;
  char* fgetPos;
  char* newline;
  char* fgetsRet;
  char* buf;

  fgetPos = buf = linebuffer->buffer;
  fgetSize = linebuffer->size;

  if (feof (stream) || ferror (stream))
    return NULL;

  while(1)
    {
      memset (fgetPos, '\1', fgetSize); /* Can be anything except 0 */
      if ((fgetsRet=fgets (fgetPos, (int) fgetSize, stream)))
	{
	  newline=memchr (fgetPos, '\n', fgetSize);
	  if (!newline) /* No \n => buffer possibly full */
	    {
	      if (buf[linebuffer->size-1] == '\0') /*buffer full*/
		{
		  if (!feof (stream))
		    {
		      fgetSize = linebuffer->size + 1;
		      linebuffer->size *= 2;
		      buf = (char *) realloc (buf, linebuffer->size);
		      linebuffer->buffer = buf;
		      fgetPos = buf + fgetSize -2;
		      continue;
		    }
		  else /* small chance incomplete line fit buffer exactly */
		    {
		      newline = &buf[linebuffer->size-1];
		      *newline = '\n';
		    }
		}
	      else /* incomplete line received */
		{
		  newline=memrchr (fgetPos, '\0', fgetSize);
		  *newline = '\n';
		}
	    }
	}
      else
	return NULL;
      break;
    }
  linebuffer->length = newline - buf + 1;
#endif
  return linebuffer;
}

/* Read a long line of text from STREAM into LINEBUFFER.
   Remove any newline & null terminate (so can't be \0's in stream),
   Return LINEBUFFER, except at end of file or error return NULL.
   Note the length is not returned in linebuffer->length but can
   be easily determined using strlen(linebuffer->buffer) if required. */
struct linebuffer *
readlineNoNulls (struct linebuffer *linebuffer, FILE *stream)
{
  size_t fgetSize;
  char* fgetPos;
  char* newline;
  char* fgetsRet;
  char* buf;

  fgetPos = buf = linebuffer->buffer;
  fgetSize = linebuffer->size;
  buf[0]='\0';

  if (feof (stream) || ferror (stream))
    return NULL;

  while(1) {
    buf[linebuffer->size-2]='\0'; //flag for buffer full
    if ((fgetsRet=fgets (fgetPos, (int) fgetSize, stream))) {
      newline=strchr (fgetPos, '\n');
      if (newline)
	*newline='\0';
      else {
	if (buf[linebuffer->size-2] != '\0') { //buffer possibly full
	  if (!feof (stream)) {
	    fgetSize = linebuffer->size + 1;
	    linebuffer->size *= 2;
	    buf = (char *) realloc (buf, linebuffer->size);
	    linebuffer->buffer = buf;
	    fgetPos = buf + fgetSize -2;
	    continue;
	  }
	  /*else small chance occurred that data matched buff size exactly*/
	}
      }
    } else {
      return NULL;
    }
    break;
  }
  return linebuffer;
}

/* Free linebuffer LINEBUFFER and its data, all allocated with malloc. */

void
freebuffer (struct linebuffer *linebuffer)
{
  free (linebuffer->buffer);
  free (linebuffer);
}
